// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import SwiftUI
import MSSImageScanner

struct UIImagePickerSampleView: UIViewControllerRepresentable {
    @Binding var isPresented: Bool
    @Binding var scanResult: ScanResult?

    func makeUIViewController(context: Context) -> UIImagePickerController {
        let imagePicker = UIImagePickerController()
        imagePicker.sourceType = .camera
        imagePicker.allowsEditing = true
        imagePicker.delegate = context.coordinator
        return imagePicker
    }

    func updateUIViewController(_ uiViewController: UIImagePickerController, context: Context) {
        // Update state
    }

    func makeCoordinator() -> SystemCameraCoordinator {
        SystemCameraCoordinator(self)
    }
}

class SystemCameraCoordinator: NSObject, UIImagePickerControllerDelegate, UINavigationControllerDelegate {
    var parent: UIImagePickerSampleView

    init(_ parent: UIImagePickerSampleView) {
        self.parent = parent
    }
    
    func imagePickerController(_ picker: UIImagePickerController, didFinishPickingMediaWithInfo info: [UIImagePickerController.InfoKey: Any]) {
        picker.dismiss(animated: true)

        guard var image = info[.editedImage] as? UIImage else {
            parent.scanResult = .cancelled
            return
        }

        image = scaleImage(image)
        decodeImage(image)
        parent.isPresented = false
    }

    func imagePickerControllerDidCancel(_ picker: UIImagePickerController) {
        picker.dismiss(animated: true)
        parent.isPresented = false
        parent.scanResult = .cancelled
    }
    
    /// Scaling image to fit SDK requirements
    private func scaleImage(_ image: UIImage) -> UIImage {
        let requiredMaxImageSize = CGFloat(ScannerConstants.imageMaxSize)
        let imageBiggerSize = max(image.size.width, image.size.height)

        if imageBiggerSize <= requiredMaxImageSize {
            // no need to scale if size requirement is fulfilled
            return image
        }

        let scale = imageBiggerSize / requiredMaxImageSize
        return UIImage(cgImage: image.cgImage!, scale: scale, orientation: image.imageOrientation)
    }

    /// Actual `UIImage` decoding stage
    private func decodeImage(_ image: UIImage) {
        do {
            let output = try QRCodeScannerSDK.decodeImage(image, codeType: .all)
            parent.scanResult = .scanned(output.result, codeType: output.codeType)
        } catch let error {
            parent.scanResult = .error(error)
        }
    }
}
